/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 1995-2004 Rohde & Schwarz, Munich
** MODULE:       UdpExample.cpp
** ABBREVIATION:   
** COMPILER:     Visual C++ 5.0
** LANGUAGE:     C/C++
** AUTHOR:       Martin Hisch
** ABSTRACT:     Example for receiving EB200/ESMB UDP packets
** PREMISES:     
** REMARKS:      
** HISTORY:
**  2011-05-13: (Mue) Version 5.19: Fixed bug: support changing port number now.
**                  Field Strength invalid flag
**  2010-11-17: (Ku) Version 5.18: Ausgabe des PIFPAN - Modes (Histogram=1; Pulse=2)
**	2005-10-5: (Hh)	Jetzt auch endlich fuer EM510
**	2005-07-05: (Mue) Version 3.91 Auch EM550
**	2005-06-30: (Ob) Version 3.91 Added WAV File support (Audiorecording)
**	2004-10-6: (Mue) Version 3.90 Ausgabe nur Channels/sec mit Parameter -c
**	2004-2-19: (Hh)	 Version 3.8: Added VideoPan and PSCAN support
**  2003-09-11: (FB) Version 3.71: Added WAV file support
**  2003-07-23: (Mue)Audio:MODE mit "E" ; CW ist nicht abhaengig von Option isCM
**          Version 3.70
**  2001-01-31: (Ks) software option EB200FS, EB200CM
**  2000-05-29: (Hh) Creation
** REVIEW:       
********************************************************************************/

/* INCLUDE FILES ***************************************************************/

/* IMPORT */
#include <stdio.h>
#include <assert.h>
#include <winsock2.h>
#include <limits.h>
#include "eb200udpsock.h"
#include "CreateUdpSock.h"

/* EXPORT */

/* GLOBAL VARIABLES DEFINITION *************************************************/

/* GLOBAL CONSTANTS DEFINITION *************************************************/

/* GLOBAL DEFINES **************************************************************/

#define BUFFER_SIZE 200

#define SCPI_PORT_NUM_DEF   5555    // (default) SCPI port numver
#define TRAC_TCP_PORT_OFFS  10      // offset rel. to SCPI port
#define MAX_DDC_NUM         4       // max. receiver instances / DDCs (curr.) supported

/* LOCAL DEFINES ***************************************************************/

#define ASSERT  assert

//#define SWAP_DEF

#define HELP_NOTICE \
"Usage:\n" \
"  UDPEXAMPLE [OPTIONS] <device>\n" \
"\n" \
"  <device> Network name or IP address of receiver device.\n" \
"\n" \
"OPTIONS:\n" \
"  -p <device IP port> TCP port of receiver device, defaults to 5555.\n" \
"\n" \
"  -d <device instance> receiver instance i.e. DDC number [0, 1..4]\n" \
"        Defaults to 0.\n" \
"\n" \
"  -am <audio mode> Audio mode. Refer to receiver manual for valid options.\n" \
"        Defaults to 0 (no audio).\n" \
"\n" \
"  -af <audio file> Name of file for the audio data to be saved in .WAV format. \n" \
"        If omitted, audio data will be forwarded to sound card, if installed.\n" \
"\n" \
"  -a    Audio only mode. All other datagrams will be ignored.\n" \
"\n" \
"  -c    Channels/sec statistics output only.\n" \
"\n" \
"  -bs   Bytes/sec statistics output only.\n" \
"\n" \
"  -gps  GPS Data Stream.\n" \
"\n" \
"  -sdw  SDW Data Stream.\n" \
"\n" \
"  -v    Verbose. Output of additional information marked with '*'.\n" \
"\n" \
"  -vc   Verbose SCPI commands send to receiver device\n" \
"\n" \
"  -vv   Verbose incl. SCPI commands send to receiver device\n" \
"\n" \
"  -iqm  Activates IQ data transmission in SHORT format.\n\n" \
"  -iqml Activates IQ data transmission in LONG format.\n\n" \
"  -iqf <IQ data file> Name of file for IQ data to be saved in stereo .WAV \n" \
"        format. Recording will be abandoned if a sample rate change occurs.\n" \
"        Implies -iqm.\n\n" \
"  -iqfl <IQ data file> Name of file for IQ data to be saved in stereo .WAV \n" \
"        format. Recording will be abandoned if a sample rate change occurs.\n" \
"        Implies -iqml.\n\n" \
"  -iqfr <IQ data file> Name of file for IQ data to be saved in raw \n" \
"        format. Recording will be abandoned if a sample rate change occurs.\n" \
"        Implies -iqm.\n" \
"\n" \
"  -csf <CSV data file> Name of file for CSV data to be saved in ascii\n" \
"        format for Excel input.\n" \
"\n" \
"  -t    TCP tracing\n" \
"\n" \
"  -rp <trace IP port> Optional trace receive port.\n" \
"        No trace configuration is done - just receive/record.\n" \
"\n" \
"  -cp   <trace IP port> Optional trace configuration port.\n" \
"        No receiving/recording is done - just configuration.\n" \
"  -ci <trace IP addr> Optional trace configuration IPv4 address.\n" \
"        Requires '-cp' option ! If not given the IPv4 address of this machine\n" \
"        is used.\n" \
"  -caif <trace AMMOS IQ data format> Optional trace configuration AMMOS IQ\n" \
"        data format.\n" \
"        Requires '-cp' option ! If not given EB200 datagram format is used.\n" \
"\n" \
"  -da   Delete (all) traces. Defaults to 'UDP' traces - use '-t' for 'TCP'.\n" \
"\n" \
"\n" \
"  Press any key to terminate UDPEXAMPLE. Don't use Ctrl-C, because this\n" \
"  prevents UDPEXAMPLE from proper closing the dump files.\n"

/* LOCAL TYPES DECLARATION *****************************************************/

/* LOCAL CLASSES DECLARATION ***************************************************/

/* LOCAL VARIABLES DEFINITION **************************************************/
bool isWPU = false;
bool isWDE = false;
bool isESMD = false;
bool isNSMI = false;
bool isEM050 = false;
bool isPR100 = false;
bool isEB200 = false;
bool isDS = false;
bool isPS = false;
bool isFS = false;
bool isCM = false;
bool isSL = false;
bool isDF = false;
bool isPIFP = false;
bool bChannelsOnly = false;
bool bBytesOnly = false;
bool bVerbose = false;
bool bVerboseCmd = false;
bool bGPSOutput = false;
bool bSDWOutput = false;

/* LOCAL CONSTANTS DEFINITION **************************************************/

/* LOCAL FUNCTIONS DEFINITION **************************************************/

/* FUNCTION ********************************************************************/
int MySend(int sd, char *pBuffer)
/*
SPECIFICATION:
Send String to socket sd
PARAMETERS:
int sd : socket descriptor
char *pBuffer : Pointer to String
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  if ( bVerboseCmd ) puts( pBuffer );

  unsigned int nLen;
  nLen = send(sd, pBuffer, strlen(pBuffer), 0);
  if (nLen != strlen(pBuffer))
  {
    printf("Error writing to socket. Len = %d\n", nLen);
  }
  return nLen;
}

/* GLOBAL FUNCTIONS DEFINITION *************************************************/

/* FUNCTION ********************************************************************/
int main(int argc, char **argv)
/*
SPECIFICATION:  
  main function for this example
PARAMETERS:
  int argc    : Number of command line arguments
  char **argv : Pointer to command line arguments
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  bool bCmdLineOK = true;
  unsigned short usPort = SCPI_PORT_NUM_DEF;
  unsigned short usRecPort = 0, usCfgPort = 0;
  unsigned long ulCfgAddr = INADDR_NONE;
  unsigned unPort, unAudioMode = 0, unDDC = 0;
  char *pcWAVFile = NULL;
  bool bAudioOnlyMode = false;
  bool bIQMode = false, bAIfOnly = false;
  bool bIQModeShort = true;
  bool bCSMode = false;
  bool bRaw = false;
  bool bTracTcp = false;
  bool bDelAll = false;
  char *pcIQFile = NULL;
  char *pcCSFile = NULL;
  char *pcDeviceAddress = NULL;
  char *pIP = NULL;
  int  len;

  /* Print banner */
  printf("UDPEXAMPLE 5.19, Copyright (c) 1998-2011 Rohde&Schwarz, Munich\n\n");

  /* Process command line */
  int param;
  for (param = 1; param < argc && bCmdLineOK; ++param)
  {
    if (strcmp(argv[param], "-p") == 0)
    {
      /* Evaluate TCP port parameter */
      if (++param < argc)
      {
        unPort = atoi(argv[param]);
        usPort = (unsigned short) unPort;
        if ((unPort == 0) || (unPort > USHRT_MAX))
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-d") == 0)
    {
      /* Evaluate receiver instance / DDC number */
      if (++param < argc)
      {
        unDDC = atoi(argv[param]);
        if (unDDC > MAX_DDC_NUM) unDDC = 0; // fallback to main receiver for invalid receiver instance
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-am") == 0)
    {
      /* Evaluate audio mode parameter */
      if (++param < argc)
      {
        unAudioMode = atoi(argv[param]);
        if (unAudioMode > 12)
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-af") == 0)
    {
      /* Evaluate audio file name */
      if (++param < argc)
      {
        pcWAVFile = argv[param];
        if (pcWAVFile[0] == '\0')
          bCmdLineOK = false;
//        if (unAudioMode == 0)
//          unAudioMode = 12;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-a") == 0)
    {
      /* Evaluate audio only mode */
      bAudioOnlyMode = true;
    }
    else if (strcmp(argv[param], "-c") == 0)
    {
      /* Evaluate scan performance only */
      bChannelsOnly = true;
    }
    else if (strcmp(argv[param], "-bs") == 0)
    {
        /* Evaluate scan performance only */
        bBytesOnly = true;
    }
    else if (strcmp(argv[param], "-gps") == 0)
    {
        /* Evaluate scan performance only */
        bGPSOutput = true;
    }
    else if (strcmp(argv[param], "-sdw") == 0)
    {
        /* Evaluate SDW output only */
        bSDWOutput = true;
    }
    else if (strcmp(argv[param], "-v") == 0)
    {
      /* Verbose additional testprints */
      bVerbose = true;
    }
    else if (strcmp(argv[param], "-vc") == 0)
    {
      bVerboseCmd = true;
    }
    else if (strcmp(argv[param], "-vv") == 0)
    {
      bVerbose = bVerboseCmd = true;
    }
    else if (strcmp(argv[param], "-iqm") == 0)
    {
      /* Evaluate IQ mode */
      bIQMode = true;
      bIQModeShort = true;
    }
    else if (strcmp(argv[param], "-iqml") == 0)
    {
      /* Evaluate IQ mode */
      bIQMode = true;
      bIQModeShort = false;
    }
    else if (strcmp(argv[param], "-iqf") == 0)
    {
      /* Evaluate IQ file name */
      if (++param < argc)
      {
        pcIQFile = argv[param];
        bIQMode = true;
        bIQModeShort = true;
        if (pcIQFile[0] == '\0')
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-iqfl") == 0)
    {
      /* Evaluate IQ file name */
      if (++param < argc)
      {
        pcIQFile = argv[param];
        bIQMode = true;
        bIQModeShort = false;
        if (pcIQFile[0] == '\0')
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-iqfr") == 0)
    {
      /* Evaluate IQ file name */
      if (++param < argc)
      {
        pcIQFile = argv[param];
        bIQMode = true;
        bRaw = true;
        if (pcIQFile[0] == '\0')
          bCmdLineOK = false;

      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-csf") == 0)
    {
      /* Evaluate Comma Separated file name */
      if (++param < argc)
      {
        pcCSFile = argv[param];
        bCSMode = true;
        if (pcCSFile[0] == '\0')
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-t") == 0)
    {
      /* TCP tracing */
      bTracTcp = true;
    }
    else if (strcmp(argv[param], "-rp") == 0)
    {
      /* Trace receive port */
      if (++param < argc)
      {
        unPort = atoi(argv[param]);
        usRecPort = (unsigned short) unPort;
        if ((unPort == 0) || (unPort > USHRT_MAX))
          bCmdLineOK = false;
      }
      else
        bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-cp") == 0)
    {
       /* Trace config port */
       if (++param < argc)
       {
         unPort = atoi(argv[param]);
         usCfgPort = (unsigned short) unPort;
         if ((unPort == 0) || (unPort > USHRT_MAX))
           bCmdLineOK = false;
       }
       else
         bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-ci") == 0)
    {
       /* Trace config ip */
       if (++param < argc)
       {
         pIP = argv[param];
       }
       else
         bCmdLineOK = false;
    }
    else if (strcmp(argv[param], "-caif") == 0)
    {
       /* Trace AMMOS IF format only */
       bAIfOnly = true;
    }
    else if (strcmp(argv[param], "-da") == 0)
    {
      bDelAll = true;
    }
    else if (param == argc-1)
    {
      /* Evaluate receiver device address */
      pcDeviceAddress = argv[argc-1];
    }
    else
      bCmdLineOK = false;
  }

  // post eval consistency
  bCmdLineOK &= !(((pIP != NULL) || bAIfOnly) && (usCfgPort == 0)); // -ci and -caif require -cp
  bCmdLineOK &= !(bDelAll && (usRecPort != 0));                     // -da conflicts with '-rp'

  /* Output help text if no command line parameter given or command line faulty */
  if (!bCmdLineOK || pcDeviceAddress == NULL)
  {
    printf(HELP_NOTICE);
    return 1;
  }

  WSADATA wsaData;
  if (WSAStartup(MAKEWORD(2, 0), &wsaData) != 0)
  {
    /* Tell the user that we couldn't find a usable */
    /* WinSock DLL.                  */
    printf("Error retrieving windows socket dll.\n");
    return 1;
  }
 
  /* Confirm that the WinSock DLL supports 2.0.*/
  /* Note that if the DLL supports versions greater  */
  /* than 2.0 in addition to 2.0, it will still return */
  /* 2.0 in wVersion since that is the version we    */
  /* requested.                    */
 
  if ( LOBYTE(wsaData.wVersion) != 2 ||
      HIBYTE(wsaData.wVersion) != 0 ) {
    /* Tell the user that we couldn't find a usable */
    /* WinSock DLL.                  */
    printf("Error retrieving correct winsock version 2.0.\n");
    WSACleanup();
    return 1; 
  }
  /* The WinSock DLL is acceptable. Proceed. */

  /* Determine receiver device IP address */
  unsigned long ulRemoteAddress = inet_addr(pcDeviceAddress);
  if (ulRemoteAddress == INADDR_NONE)
  {
    hostent* phost = gethostbyname(pcDeviceAddress);
    if (phost && phost->h_addr_list[0] != NULL)
      ulRemoteAddress = *((unsigned*) phost->h_addr_list[0]);
    else
    {
      printf("Can't resolve host name %s.\n", pcDeviceAddress);
      return 1;
    }
  }

  SOCKET sock = INVALID_SOCKET;
  // if receive (only) port was not set -> setup SCPI connection for config.
  if ( (usRecPort == 0) || ((usRecPort != 0) && (usCfgPort != 0)) )
  {
     /* Create a receiver control TCP socket */
     sock = socket(AF_INET, SOCK_STREAM, 0);
     if (sock == SOCKET_ERROR)
     {
       printf("Error creating SCPI socket.\n");
       return 1;
     }

     int sopt = 1;
     if ( setsockopt( sock, IPPROTO_TCP, TCP_NODELAY, (char *)&sopt, sizeof( sopt ) ) == SOCKET_ERROR )
     {
        printf("setsockopt (TCP_NODELAY, %d) failed - errno %d\n", sopt, WSAGetLastError());
        return 1;
     }

     /* Create socket address structure for INET address family */
     struct sockaddr_in addrDevice;
     memset(&addrDevice, 0, sizeof(addrDevice));
     addrDevice.sin_family      = AF_INET;
     addrDevice.sin_addr.s_addr = ulRemoteAddress;
     addrDevice.sin_port        = htons(usPort);

     /* connect socket to receiver device */
     if (connect(sock, (struct sockaddr *)&addrDevice, sizeof(addrDevice)) != 0)
     {
       printf("Error connecting to %s [SCPI port = %d]\n", pcDeviceAddress, usPort);
       return 1;
     }
  }

  static char cBuffer[BUFFER_SIZE], *pBuffer = cBuffer;

  char cDDCx[8]; cDDCx[0] = '\0';
  if (unDDC > 0) sprintf(cDDCx, ":DDC%d", unDDC);

  char cTracType[16];
  sprintf(cTracType, "TRAC%s:%s", cDDCx, bTracTcp ? "TCP" : "UDP");

  /* if '-da' switch is set: delete all trace paths and leave */
  if ( bDelAll && (sock != INVALID_SOCKET) )
  {
    sprintf(cBuffer, "%s:DEL ALL\n", cTracType);
    MySend(sock, cBuffer);
    sprintf(cBuffer, "%s?\n", cTracType);
    MySend(sock, cBuffer);
    len = recv(sock, cBuffer, sizeof(cBuffer),0);
    if( len < 0 ) len = 0; cBuffer[ len ] = '\0';

    printf("All %s TRACes deleted:\n", bTracTcp ? "TCP" : "UDP");
    printf("%s", cBuffer);

    if (isWPU && bSDWOutput)
    {
        sprintf(cBuffer, "SYSTem:WPU:SDW:MODE OFF\n");
        MySend(sock, cBuffer);
    }

    return (0);
  }

  unPort = usRecPort;

  CEB200UdpSock *pTracSock = NULL;
  // if config. (only) port was not set -> setup TRACe connection for receive/record.
  if ( (usCfgPort == 0) || ((usRecPort != 0) && (usCfgPort != 0)) )
  {
     /* create TCP/UDP trace socket instance */
     pTracSock = CreateUdpSock( bTracTcp, bTracTcp ? usPort + TRAC_TCP_PORT_OFFS : usPort, ulRemoteAddress, unPort );

     unPort = pTracSock->GetTracPort();
     if ( pTracSock->GetTracSock() == INVALID_SOCKET )
     {
       printf("Error setting up TRACe connection to %s [TRAC port = %d]\n", pcDeviceAddress, unPort);
       return 1;
     }
     printf("TRACe connection to %s [TRAC port = %d] successful !\n", pcDeviceAddress, unPort);
  }
  else
    unPort = usCfgPort;

  /* do some TRACe configuration */
  /* retrieve local IP address in case it's not set */
  if ( pIP == NULL )
  {
    struct sockaddr_in addrLocal;
    int addrlen = sizeof(addrLocal);
    getsockname(sock, (struct sockaddr*)&addrLocal, &addrlen);
    pIP = inet_ntoa(addrLocal.sin_addr);
  }
  else
  {
    ulCfgAddr = inet_addr(pIP);
    if ( ulCfgAddr == INADDR_NONE )
    {
      hostent* phost = gethostbyname(pIP);
      if (phost && phost->h_addr_list[0] != NULL)
        ulCfgAddr = *((unsigned*) phost->h_addr_list[0]);
      else
      {
        printf("Can't resolve trace host name %s.\n", pIP);
        return 1;
      }
      struct in_addr cfgaddr;
      cfgaddr.s_addr = ulCfgAddr;
      pIP = inet_ntoa(cfgaddr);

    }
  }

  static char cTracFlag[BUFFER_SIZE];
  sprintf(cTracFlag, "%s:FLAG \"%s\", %d", cTracType, pIP, unPort);

  static char cTracTag[BUFFER_SIZE];
  sprintf(cTracTag, "%s:TAG \"%s\", %d", cTracType, pIP, unPort);

  // if SCPI connection was setup above -> setup configuration...
  if ( sock != INVALID_SOCKET )
  {
    /* If no multi-access to receiver device required delete all trace paths */
    // sprintf(cBuffer, "%s:DEL ALL\n", cTracType);
    // MySend(sock, cBuffer);

    sprintf(cBuffer, "*OPT?\n");
    MySend(sock, cBuffer);
    len = recv(sock, cBuffer, sizeof(cBuffer),0);
    if( len < 0 ) len = 0; cBuffer[ len ] = '\0';

    if (strstr(cBuffer,"DS") != NULL) {
      isDS = true;
    }
    if (strstr(cBuffer,"PS") != NULL) {
      isPS = true;
    }
    if (strstr(cBuffer,"FS") != NULL) {
      isFS = true;
    }
    if (strstr(cBuffer,"CM") != NULL) {
      isCM = true;
    }
    if (strstr(cBuffer,"SL") != NULL) {
      isSL = true;
    }
    if (strstr(cBuffer,"DF") != NULL) {
      isDF = true;
    }
    MySend(sock, "*IDN?\n");
    len = recv(sock, cBuffer, sizeof(cBuffer),0);
    if( len < 0 ) len = 0; cBuffer[ len ] = '\0';

	if (strstr(cBuffer,"EB200") != NULL) {
      isEB200 = true;
    }
	if (strstr(cBuffer,"ESMB") != NULL) {
      isEB200 = true;
    }
	if (strstr(cBuffer,"EB110") != NULL) {
      isEB200 = true;
    }

    if (strstr(cBuffer,"EM050") != NULL) {
      isEM050 = true;
    }
    if (strstr(cBuffer,"EM550") != NULL) {
      isEM050 = true;
    }
    if (strstr(cBuffer,"EM510") != NULL) {
      isEM050 = true;
    }
    if (strstr(cBuffer,"WPU") != NULL) {
      isESMD = true;
      isWPU = true;
    }
    if (strstr(cBuffer,"WDE") != NULL) {
      isWDE = true;
    }
    if (isWDE || (strstr(cBuffer,"ESMD") != NULL)) {
      isESMD = true;
      isPIFP = true;
      isFS = true;
    }
    if (strstr(cBuffer,"NSMI") != NULL)
    {
        isESMD = true;
    }
    if (strstr(cBuffer,"DDF255") != NULL) {
      isESMD = true;
      isPIFP = true;
      isFS = true;
    }
    if (strstr(cBuffer,"MBR100") != NULL) {
      isESMD = true;
      isFS = true;
    }
    if (strstr(cBuffer,"EB500") != NULL
        || strstr(cBuffer,"EB510") != NULL
        || strstr(cBuffer,"DDF205") != NULL) {
      isESMD = true;
      isPIFP = true;
      isFS = true;
    }
    if (   strstr(cBuffer,"PR100") != NULL 
        || strstr(cBuffer,"EM100") != NULL) {
      isPR100 = true;
      // isFS = true;
    }

    sprintf(cBuffer, "FREQ? MAX\n");
    MySend(sock, cBuffer);
    len = recv(sock, cBuffer, sizeof(cBuffer),0);
    if( len < 0 ) len = 0; cBuffer[ len ] = '\0';

    double lfFmax = atof(cBuffer);

    /* Check if trace path available */
  /*
    int status;
    sprintf(cBuffer, "%s, FSC\n", cTracTag);
    MySend(sock, cBuffer);
    MySend(sock, "SYST:ERR?");
    MyRecv(sock, cBuffer, 1);
    sscanf(cBuffer, "%d", &status);
    if (status != 0)
    {
      printf("No trace available in receiver device: %s", cBuffer);
      return 1;
    }
    sprintf(cBuffer, "%s:TAG:OFF \"%s\", %d, FSC\n", cTracType, pIP, unPort);
    MySend(sock, cBuffer);
  */
    if (bChannelsOnly)
    {
        /* disable FStrength and DF option */
        isDF = false;
        isFS = false;
    }
    /* Configure receiver device traces */

    if (!bAudioOnlyMode && !bAIfOnly)
    {
      sprintf(cBuffer, "%s, FSC, MSC, DSC, AUD, IFP, CW\n", cTracTag);
      MySend(sock, cBuffer);
      if (isCM)
      {
          sprintf(cBuffer, "%s, FAST, LIST\n", cTracTag);
          MySend(sock, cBuffer);
      }
      if (isEM050 || isESMD)
      {
          if (isSL)
          {
              sprintf(cBuffer, "%s, IF, VID, VDP, PSC, SELC\n", cTracTag);
          }
          else
          {
              sprintf(cBuffer, "%s, IF, VID, VDP, PSC\n", cTracTag);
          }

          MySend(sock, cBuffer);
          if (isDF)
          {
              sprintf(cBuffer, "%s, DFP\n", cTracTag);
              MySend(sock, cBuffer);
          }
          if (isPIFP)
          {
              sprintf(cBuffer, "%s, PIFP\n", cTracTag);
              MySend(sock, cBuffer);
          }
          if (isWDE)
          {
              sprintf(cBuffer, "%s, RDP\n", cTracTag);
              MySend(sock, cBuffer);
              sprintf(cBuffer, "%s, SCD\n", cTracTag);
              MySend(sock, cBuffer);
          }
      }
      if (isESMD)
      {
          if(bGPSOutput)
          {
              sprintf(cBuffer, "%s, GPSC\n", cTracTag);
              MySend(sock, cBuffer);
          }
      }
      if (isPR100)
      {
          sprintf(cBuffer, "%s, IF, PSC\n", cTracTag);

          MySend(sock, cBuffer);
          if(bGPSOutput)
          {
              sprintf(cBuffer, "%s, GPSC\n", cTracTag);
              MySend(sock, cBuffer);
          }
      }
      sprintf(cBuffer, "%s, \"VOLT:AC\", \"FREQ:RX\", \"FREQ:OFFS\", \"OPT\"\n", cTracFlag);
      MySend(sock, cBuffer);
      if ( isWDE ){
        sprintf(cBuffer, "%s, \"FREQ:HIGH:RX\", \"CHAN\", \"WDE:THR\", \"WDE:FLAG\" \n", cTracFlag);
        MySend(sock, cBuffer);
      }
      if (isFS || isESMD) {
        sprintf(cBuffer, "%s, \"FSTRength\"\n", cTracFlag);
        MySend(sock, cBuffer);
        // and also set up the appropriate sensor function
        MySend(sock, "SENS:FUNC 'FSTR'\n");
      }
      if (isDF) {
        sprintf(cBuffer, "%s, \"AZIM\", \"DFQ\", \"DFL\"\n", cTracFlag);
        MySend(sock, cBuffer);
        if (isFS) {
        sprintf(cBuffer, "%s, \"DFFSTRength\"\n", cTracFlag);
        MySend(sock, cBuffer);
        }
      }
      if (isPR100) 
      {
        sprintf(cBuffer, "%s, \"FREQ:LOW:RX\"\n", cTracFlag);
        MySend(sock, cBuffer);
      }
      if (lfFmax > 4.29E9)
      {
        sprintf(cBuffer, "%s, \"FREQ:HIGH:RX\"\n", cTracFlag);
        MySend(sock, cBuffer);
      }

  #ifdef SWAP_DEF
      sprintf(cBuffer, "%s, \"SWAP\"\n", cTracFlag);
      MySend(sock, cBuffer);
  #endif
    }
    else if ( bAIfOnly )
    {
      sprintf(cBuffer, "%s, AIF\n", cTracTag);
      MySend(sock, cBuffer);
      sprintf(cBuffer, "%s, \"OPT\"\n", cTracFlag);
      MySend(sock, cBuffer);
    }
    else
    {
      sprintf(cBuffer, "%s, AUDIO\n", cTracTag);
      MySend(sock, cBuffer);
      sprintf(cBuffer, "%s, \"OPT\"\n", cTracFlag);
      MySend(sock, cBuffer);
    }
    if (isWPU && bSDWOutput)
    {
        /* get only SDW trace */
        sprintf(cBuffer, "%s, SDW\n", cTracTag);
        MySend(sock, cBuffer);

        /* enable SDW flag */
        sprintf(cBuffer, "%s, \"OPT\", \"SDW\" \n", cTracFlag);
        MySend(sock, cBuffer);
    }

    /* configure audio mode */
    sprintf(cBuffer, "SYST:AUD%s:REM:MODE %d\n", cDDCx, unAudioMode);
    MySend(sock, cBuffer);

    if (isEM050 || isESMD)
    {
        /* configure IQ data mode */
        char *pBuffer = cBuffer;
        pBuffer += sprintf(pBuffer, "SYST:IF%s:REM:MODE ", cDDCx);

        if ( !bAIfOnly )
        {
            if (bIQModeShort)
            {
                pBuffer += sprintf(pBuffer, bIQMode? "SHORT" : "OFF");
            }
            else
            {
                pBuffer += sprintf(pBuffer, bIQMode? "LONG" : "OFF");
            }
        }
        else
        {
           pBuffer += sprintf(pBuffer, "ASHORT");
        }

        sprintf(pBuffer, "\n");
        MySend(sock, cBuffer);
    }

    /* enable SDW output */
    if (isWPU && bSDWOutput)
    {
        sprintf(cBuffer, "SYSTem:WPU:SDW:MODE ON\n");
        MySend(sock, cBuffer);
    }

    /* finally check for errors */
    sprintf(cBuffer, "SYST:ERR%s?\n", isEB200 ? "" : ":ALL");   // EB200 doesn't know SYST:ERR:ALL);
    MySend(sock, cBuffer);
    len = recv(sock, cBuffer, sizeof(cBuffer),0);
    if( len < 0 ) len = 0; cBuffer[ len ] = '\0';
    puts( cBuffer );
  }

  // if tracing is used...
  if ( pTracSock != NULL )
  {
    /* IQ File Recording? */
    if ( pcIQFile )  pTracSock->SetIFRecording(pcIQFile, bRaw);

    /* CS File Recording? */
    if ( pcCSFile )  pTracSock->SetCSRecording(pcCSFile);

    /* WAV File Recording? */
    if ( pcWAVFile ) pTracSock->SetAFRecording(pcWAVFile);

    /* call socket reception routine */
    pTracSock->Init();

    if ( sock != INVALID_SOCKET )
    {
      /* stop data transfer from receiver device */
      sprintf(cBuffer, "%s:DEL \"%s\", %d\n", cTracType, pIP, unPort);
      MySend(sock, cBuffer);

      /* stop SDW generation if enabled */
      if (isWPU && bSDWOutput)
      {
          sprintf(cBuffer, "SYSTem:WPU:SDW:MODE OFF\n");
          MySend(sock, cBuffer);
      }

      /* finally check for errors merely to sync to the TRAC:<UDP|TCP>:DEL command processing above */
      /* This is more robust than relying on the (optional) socket shutdown mechanism. */

		if(isEB200) //EB200 doesn't know SYST:ERR:ALL?
		{
			MySend(sock, "SYST:ERR?\n");
			len = recv(sock, cBuffer, sizeof(cBuffer),0);
			if( len < 0 ) len = 0; cBuffer[ len ] = '\0';
			puts( cBuffer );
		}

		else
		{
		  MySend(sock, "SYST:ERR:ALL?\n");
		  len = recv(sock, cBuffer, sizeof(cBuffer),0);
		  if( len < 0 ) len = 0; cBuffer[ len ] = '\0';
		  puts( cBuffer );
		}

    }
  }
  return 0;
}

/* FUNCTION ********************************************************************/
bool EquipmentisEM050(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(isEM050 || isESMD);
}

/* FUNCTION ********************************************************************/
bool OptionisDS(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(isDS);
}

/* FUNCTION ********************************************************************/
bool OptionisPS(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(isPS);
}

/* FUNCTION ********************************************************************/
bool OptionisFS(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(isFS);
}

/* FUNCTION ********************************************************************/
bool OptionisCM(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(isCM);
}

/* FUNCTION ********************************************************************/
bool ChannelsOnly(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(bChannelsOnly);
}
/* FUNCTION ********************************************************************/
bool BytesOnly(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
    return(bBytesOnly);
}
/* FUNCTION ********************************************************************/
bool Verbose(void)
/*
SPECIFICATION:
PARAMETERS:
void : 
PRECONDITIONS: 
SIDE_EFFECTS: 
RETURN_VALUES: 
EXCEPTIONS: 
********************************************************************************/
{
  return(bVerbose);
}

